/*
goredo -- djb's redo implementation on pure Go
Copyright (C) 2020-2023 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

package main

import (
	"flag"
	"fmt"
	"os"
)

const (
	Version  = "2.4.0"
	Warranty = `Copyright (C) 2020-2023 Sergey Matveev

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.`
)

func usage(cmd string) {
	var d string
	switch cmd {
	case CmdNameRedo:
		d = `Usage: redo [-j X] [-k] [-s] [-x|-xx] [options] [target ...]

Forcefully and *sequentially* build specified targets.
If no targets specified, then use "all" one.`
	case CmdNameRedoIfchange:
		d = `Usage: redo-ifchange [-j X] [-k] [-s] [-x|-xx] target [...]

Build specified targets in parallel, if they are changed.
Record them as dependencies for current target.`
	case CmdNameRedoIfcreate:
		d = `Usage: redo-ifcreate target [...]

Record ifcreate dependency for current target. Unusable outside .do.`
	case CmdNameRedoAlways:
		d = `Usage: redo-always

Always build current target. Unusable outside .do.`
	case CmdNameRedoCleanup:
		d = `Usage: redo-cleanup [-n] {full,log,lock,tmp} [...]

Remove either all of goredo's related temporary files, or kept stderr
logs, or lock files, or everything (including .redo directories) related.`
	case CmdNameRedoLog:
		d = `Usage: redo-log [-c] [-r] target [ | tai64nlocal ]

Display kept target's stderr with TAI64N timestamped lines. Only the
last build is kept. You must enable stderr keeping with either -logs,
or REDO_LOGS=1. -c option show the exact command invoked, start and
finish time. -r option recursively and linearly show also all deeper
redo calls and their logs.`
	case CmdNameRedoDot:
		d = `Usage: redo-dot target [...]

Write dependency DOT graph to stdout.`
	case CmdNameRedoStamp:
		d = `Usage: redo-stamp < [$3]

Record stamp dependency for current target. Unusable outside .do.
Stamp dependency does not play any role, as all targets are hashed
anyway.`
	case CmdNameRedoWhichdo:
		d = `Usage: redo-whichdo target

Display .do search paths for specified target. Exits successfully
if the last .do in output if the found existing one.`
	case CmdNameRedoTargets:
		d = `Usage: redo-targets [target ...]

List all currently known targets.`
	case CmdNameRedoSources:
		d = `Usage: redo-sources [target ...]

List all currently known source files.`
	case CmdNameRedoOOD:
		d = `Usage: redo-ood [target ...]

List all currently known out-of-date targets.`
	case CmdNameRedoAffects:
		d = `Usage: redo-affects target [...]

List all targets that will be affected by changing the specified ones.`
	case CmdNameRedoDep2Rec:
		d = `Usage: redo-dep2rec .../.redo/file.dep

Convert binary .dep file to recfile and write it to stdout.`
	case CmdNameRedoDepFix:
		d = `Usage: redo-depfix

Traverse over all .redo directories beneath and recalculate each target's
inode and hash information, rewriting dependency files. If dependency has
legacy .rec format, then it will be converted to .dep one.`
	case CmdNameRedoInode:
		d = `Usage: redo-inode target [...]

Just calculate inode information about each target and print in recfile format.`
	default:
		d = `Usage: goredo -symlinks

goredo expects to be called through the symbolic link to it.
Available commands: redo, redo-affects, redo-always, redo-cleanup,
redo-dep2rec, redo-depfix, redo-dot, redo-ifchange, redo-ifcreate,
redo-inode, redo-log, redo-ood, redo-sources, redo-stamp,
redo-targets, redo-whichdo.`
	}
	fmt.Fprintf(os.Stderr, "%s\n\nCommon options:\n", d)
	flag.PrintDefaults()
	fmt.Fprintln(os.Stderr, `
Additional environment variables:
  NO_COLOR -- disable messages colouring
  REDO_TOP_DIR -- do not search for .do above that directory
                  (it can contain .redo/top as an alternative)`)
	if cmd == CmdNameRedo || cmd == CmdNameRedoIfchange {
		fmt.Fprintln(os.Stderr, `
  REDO_NO_SYNC -- disable files/directories explicit filesystem syncing
  REDO_INODE_TRUST -- {none,ctime,mtime}, either do not trust inode
                      information at all (always check size and hash), or
                      trust its ctime (the default one), or be satisfied
                      with its mtime
  REDO_MAKE -- bmake/gmake/none(default) jobserver protocol compatibility`)
	}
}
