/*
goredo -- djb's redo implementation on pure Go
Copyright (C) 2020-2021 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// Targets runner

package main

import (
	"bufio"
	"errors"
	"flag"
	"fmt"
	"io"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"strings"
	"sync"
	"syscall"
	"time"

	"go.cypherpunks.ru/recfile"
	"go.cypherpunks.ru/tai64n/v2"
	"golang.org/x/sys/unix"
)

const (
	EnvDepFd        = "REDO_DEP_FD"
	EnvDirPrefix    = "REDO_DIRPREFIX"
	EnvBuildUUID    = "REDO_BUILD_UUID"
	EnvStderrPrefix = "REDO_STDERR_PREFIX"
	EnvTrace        = "REDO_TRACE"
	EnvStderrKeep   = "REDO_LOGS"
	EnvStderrSilent = "REDO_SILENT"
	EnvNoSync       = "REDO_NO_SYNC"

	RedoDir    = ".redo"
	LockSuffix = ".lock"
	DepSuffix  = ".rec"
	TmpPrefix  = ".redo."
	LogSuffix  = ".log"
)

var (
	NoSync       bool = false
	StderrKeep   bool = false
	StderrSilent bool = false
	StderrPrefix string
	Jobs         sync.WaitGroup

	flagTrace        = flag.Bool("x", false, "trace (sh -x) current targets")
	flagTraceAll     = flag.Bool("xx", false, fmt.Sprintf("trace (sh -x) all targets (%s=1)", EnvTrace))
	flagStderrKeep   = flag.Bool("logs", false, fmt.Sprintf("keep job's stderr (%s=1)", EnvStderrKeep))
	flagStderrSilent = flag.Bool("silent", false, fmt.Sprintf("do not print job's stderr (%s=1)", EnvStderrSilent))

	TracedAll bool
)

type RunErr struct {
	Tgt      string
	DoFile   string
	Started  *time.Time
	Finished *time.Time
	Err      error
}

func (e *RunErr) Name() string {
	var name string
	if e.DoFile == "" {
		name = e.Tgt
	} else {
		name = fmt.Sprintf("%s (%s)", e.Tgt, e.DoFile)
	}
	if e.Finished == nil {
		return name
	}
	return fmt.Sprintf("%s (%fsec)", name, e.Finished.Sub(*e.Started).Seconds())
}

func (e RunErr) Error() string {
	return fmt.Sprintf("%s: %s", e.Name(), e.Err)
}

func mkdirs(pth string) error {
	if _, err := os.Stat(pth); err == nil {
		return nil
	}
	return os.MkdirAll(pth, os.FileMode(0777))
}

func isModified(cwd, redoDir, tgt string) (bool, *Inode, error) {
	fdDep, err := os.Open(path.Join(redoDir, tgt+DepSuffix))
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil, nil
		}
		return false, nil, err
	}
	defer fdDep.Close()
	r := recfile.NewReader(fdDep)
	var ourInode *Inode
	for {
		m, err := r.NextMap()
		if err != nil {
			if err == io.EOF {
				break
			}
			return false, nil, err
		}
		if m["Target"] != tgt {
			continue
		}
		fd, err := os.Open(path.Join(cwd, tgt))
		if err != nil {
			if os.IsNotExist(err) {
				return false, nil, nil
			}
			return false, nil, err
		}
		ourInode, err = inodeFromFile(fd)
		fd.Close()
		if err != nil {
			return false, nil, err
		}
		theirInode, err := inodeFromRec(m)
		if err != nil {
			return false, nil, err
		}
		if !ourInode.Equals(theirInode) {
			return true, ourInode, nil
		}
		break
	}
	return false, ourInode, nil
}

func syncDir(dir string) error {
	fd, err := os.Open(dir)
	if err != nil {
		return err
	}
	err = fd.Sync()
	fd.Close()
	return err
}

func runScript(tgtOrig string, errs chan error, traced bool) error {
	cwd, tgt := cwdAndTgt(tgtOrig)
	redoDir := path.Join(cwd, RedoDir)
	if err := mkdirs(redoDir); err != nil {
		return TgtErr{tgtOrig, err}
	}

	// Acquire lock
	fdLock, err := os.OpenFile(
		path.Join(redoDir, tgt+LockSuffix),
		os.O_WRONLY|os.O_TRUNC|os.O_CREATE,
		os.FileMode(0666),
	)
	if err != nil {
		return TgtErr{tgtOrig, err}
	}
	lockRelease := func() {
		trace(CLock, "LOCK_UN: %s", fdLock.Name())
		unix.Flock(int(fdLock.Fd()), unix.LOCK_UN)
		fdLock.Close()
	}
	trace(CLock, "LOCK_NB: %s", fdLock.Name())

	// Waiting for job completion, already taken by someone else
	if err = unix.Flock(int(fdLock.Fd()), unix.LOCK_EX|unix.LOCK_NB); err != nil {
		if uintptr(err.(syscall.Errno)) != uintptr(unix.EWOULDBLOCK) {
			fdLock.Close()
			return TgtErr{tgtOrig, err}
		}
		Jobs.Add(1)
		trace(CDebug, "waiting: %s", tgtOrig)
		if FdStatus != nil {
			FdStatus.Write([]byte{StatusWait})
		}
		go func() {
			defer Jobs.Done()
			trace(CLock, "LOCK_EX: %s", fdLock.Name())
			unix.Flock(int(fdLock.Fd()), unix.LOCK_EX)
			lockRelease()
			trace(CDebug, "waiting done: %s", tgtOrig)
			if FdStatus != nil {
				FdStatus.Write([]byte{StatusWaited})
			}
			var depInfo *DepInfo
			fdDep, err := os.Open(path.Join(redoDir, tgt+DepSuffix))
			if err != nil {
				if os.IsNotExist(err) {
					err = errors.New("was not built: no .rec")
				}
				goto Finish
			}
			defer fdDep.Close()
			depInfo, err = depRead(fdDep)
			if err != nil {
				goto Finish
			}
			if depInfo.build != BuildUUID {
				err = errors.New("was not built: build differs")
			}
		Finish:
			if err != nil {
				err = TgtErr{tgtOrig, err}
			}
			errs <- err
		}()
		return nil
	}

	// Check if target is not modified externally
	modified, inodePrev, err := isModified(cwd, redoDir, tgt)
	if err != nil {
		lockRelease()
		return TgtErr{tgtOrig, err}
	}
	if modified {
		trace(CWarn, "%s externally modified: not redoing", tgtOrig)
		lockRelease()
		go func() {
			errs <- nil
		}()
		return nil
	}

	// Start preparing .rec
	fdDep, err := tempfile(redoDir, tgt+DepSuffix)
	if err != nil {
		lockRelease()
		return TgtErr{tgtOrig, err}
	}
	fdDepPath := fdDep.Name()
	cleanup := func() {
		lockRelease()
		fdDep.Close()
		os.Remove(fdDep.Name())
	}
	if _, err = recfile.NewWriter(fdDep).WriteFields(
		recfile.Field{Name: "Build", Value: BuildUUID},
	); err != nil {
		cleanup()
		return TgtErr{tgtOrig, err}
	}

	// Find .do
	doFile, upLevels, err := findDo(fdDep, cwd, tgt)
	if err != nil {
		cleanup()
		return TgtErr{tgtOrig, err}
	}
	if doFile == "" {
		cleanup()
		return TgtErr{tgtOrig, errors.New("no .do found")}
	}

	// Determine basename and DIRPREFIX
	doFileRelPath := doFile
	ents := strings.Split(cwd, "/")
	ents = ents[len(ents)-upLevels:]
	dirPrefix := path.Join(ents...)
	cwdOrig := cwd
	for i := 0; i < upLevels; i++ {
		cwd = path.Join(cwd, "..")
		doFileRelPath = path.Join("..", doFileRelPath)
	}
	cwd = path.Clean(cwd)
	doFilePath := path.Join(cwd, doFile)
	basename := tgt
	runErr := RunErr{Tgt: tgtOrig}
	if strings.HasPrefix(doFile, "default.") {
		basename = tgt[:len(tgt)-(len(doFile)-len("default.")-len(".do"))-1]
		runErr.DoFile = doFileRelPath
	}

	if err = writeDep(fdDep, cwdOrig, doFileRelPath); err != nil {
		cleanup()
		return TgtErr{tgtOrig, err}
	}
	fdDep.Close()
	trace(CWait, "%s", runErr.Name())

	// Prepare command line
	var cmdName string
	var args []string
	if err = unix.Access(doFilePath, unix.X_OK); err == nil {
		cmdName = doFilePath
		args = make([]string, 0, 3)
	} else {
		cmdName = "/bin/sh"
		if traced || TracedAll {
			args = append(args, "-ex")
		} else {
			args = append(args, "-e")
		}
		args = append(args, doFile)
	}

	// Temporary file for stdout
	fdStdout, err := tempfile(cwdOrig, tgt)
	if err != nil {
		cleanup()
		return TgtErr{tgtOrig, err}
	}
	stdoutPath := fdStdout.Name()
	fdStdout.Close()
	tmpPath := stdoutPath + ".3" // and for $3
	tmpPathRel, err := filepath.Rel(cwd, tmpPath)
	if err != nil {
		panic(err)
	}
	args = append(
		args,
		path.Join(dirPrefix, tgt),
		path.Join(dirPrefix, basename),
		tmpPathRel,
	)

	cmd := exec.Command(cmdName, args...)
	cmd.Dir = cwd
	// cmd.Stdin reads from /dev/null by default
	cmd.Env = append(os.Environ(), fmt.Sprintf("%s=%d", EnvLevel, Level+1))
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", EnvDirPrefix, dirPrefix))
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", EnvBuildUUID, BuildUUID))

	childStderrPrefix := tempsuffix()
	cmd.Env = append(cmd.Env, fmt.Sprintf(
		"%s=%s", EnvStderrPrefix, childStderrPrefix,
	))

	fdNum := 0
	cmd.ExtraFiles = append(cmd.ExtraFiles, FdOODTgts)
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvOODTgtsFd, 3+fdNum))
	fdNum++
	cmd.ExtraFiles = append(cmd.ExtraFiles, FdOODTgtsLock)
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvOODTgtsLockFd, 3+fdNum))
	fdNum++

	if FdStatus == nil {
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=NO", EnvStatusFd))
	} else {
		cmd.ExtraFiles = append(cmd.ExtraFiles, FdStatus)
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvStatusFd, 3+fdNum))
		fdNum++
	}

	if JSR == nil {
		// infinite jobs
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=NO", EnvJSFd))
	} else {
		cmd.ExtraFiles = append(cmd.ExtraFiles, JSR)
		cmd.ExtraFiles = append(cmd.ExtraFiles, JSW)
		cmd.Env = append(cmd.Env, fmt.Sprintf(
			"%s=%d,%d", EnvJSFd, 3+fdNum+0, 3+fdNum+1,
		))
		fdNum += 2
	}

	// Preparing stderr
	var fdStderr *os.File
	if StderrKeep {
		fdStderr, err = os.OpenFile(
			path.Join(redoDir, tgt+LogSuffix),
			os.O_WRONLY|os.O_CREATE,
			os.FileMode(0666),
		)
		if err != nil {
			cleanup()
			return TgtErr{tgtOrig, err}
		}
		fdStderr.Truncate(0)
	}
	shCtx := fmt.Sprintf(
		"sh: %s: %s %s cwd:%s dirprefix:%s",
		tgtOrig, cmdName, args, cwd, dirPrefix,
	)
	trace(CDebug, "%s", shCtx)

	Jobs.Add(1)
	go func() {
		jsAcquire(shCtx)
		if FdStatus != nil {
			FdStatus.Write([]byte{StatusRun})
		}

		started := time.Now()
		runErr.Started = &started
		fdStdout, err = os.OpenFile(stdoutPath, os.O_RDWR, os.FileMode(0666))
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		cmd.Stdout = fdStdout
		fdDep, err = os.OpenFile(fdDepPath, os.O_WRONLY|os.O_APPEND, os.FileMode(0666))
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		cmd.ExtraFiles = append(cmd.ExtraFiles, fdDep)
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvDepFd, 3+fdNum))
		fdNum++

		defer func() {
			jsRelease(shCtx)
			lockRelease()
			fdDep.Close()
			fdStdout.Close()
			if fdStderr != nil {
				fdStderr.Close()
			}
			os.Remove(fdDep.Name())
			os.Remove(fdStdout.Name())
			os.Remove(tmpPath)
			os.Remove(fdLock.Name())
			if FdStatus != nil {
				FdStatus.Write([]byte{StatusDone})
			}
			Jobs.Done()
		}()
		stderr, err := cmd.StderrPipe()
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		started = time.Now()
		err = cmd.Start()
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		pid := fmt.Sprintf("[%d]", cmd.Process.Pid)
		trace(CDebug, "%s runs %s", tgtOrig, pid)

		stderrTerm := make(chan struct{}, 0)
		go func() {
			scanner := bufio.NewScanner(stderr)
			var line string
			ts := new(tai64n.TAI64N)
			for scanner.Scan() {
				line = scanner.Text()
				if strings.HasPrefix(line, childStderrPrefix) {
					line = line[len(childStderrPrefix):]
					os.Stderr.WriteString(StderrPrefix + line + "\n")
					continue
				}
				if fdStderr != nil {
					ts.FromTime(time.Now())
					LogMutex.Lock()
					fmt.Fprintln(fdStderr, tai64n.Encode(ts[:]), line)
					LogMutex.Unlock()
				}
				if StderrSilent {
					continue
				}
				if MyPid == 0 {
					trace(CNone, "%s", line)
				} else {
					trace(CNone, "%s %s", pid, line)
				}
			}
			close(stderrTerm)
		}()

		// Wait for job completion
		<-stderrTerm
		err = cmd.Wait()
		finished := time.Now()
		runErr.Finished = &finished
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}

		// Was $1 touched?
		if inodePrev != nil {
			if fd, err := os.Open(path.Join(cwdOrig, tgt)); err == nil {
				inode, err := inodeFromFile(fd)
				fd.Close()
				if err == nil && !inode.Equals(inodePrev) {
					runErr.Err = errors.New("$1 was explicitly touched")
					errs <- runErr
					return
				}
			}
		}

		// Does it produce both stdout and tmp?
		fiStdout, err := os.Stat(fdStdout.Name())
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		tmpExists := false
		_, err = os.Stat(tmpPath)
		if err == nil {
			if fiStdout.Size() > 0 {
				runErr.Err = errors.New("created both tmp and stdout")
				errs <- runErr
				return
			}
			tmpExists = true
		} else if !os.IsNotExist(err) {
			runErr.Err = err
			errs <- runErr
			return
		}

		// Determine what file we must process at last
		var fd *os.File
		if tmpExists {
			fd, err = os.Open(tmpPath)
			if err != nil {
				goto Finish
			}
			defer fd.Close()
		} else if fiStdout.Size() > 0 {
			fd = fdStdout
		}

		// Do we need to ifcreate it, of ifchange with renaming?
		if fd == nil {
			os.Remove(path.Join(cwdOrig, tgt))
			err = ifcreate(fdDep, tgt)
			if err != nil {
				goto Finish
			}
		} else {
			if !NoSync {
				err = fd.Sync()
				if err != nil {
					goto Finish
				}
			}
			err = os.Rename(fd.Name(), path.Join(cwdOrig, tgt))
			if err != nil {
				goto Finish
			}
			if !NoSync {
				err = syncDir(cwdOrig)
				if err != nil {
					goto Finish
				}
			}
			err = writeDep(fdDep, cwdOrig, tgt)
			if err != nil {
				goto Finish
			}
		}

		// Commit .rec
		if !NoSync {
			err = fdDep.Sync()
			if err != nil {
				goto Finish
			}
		}
		err = os.Rename(fdDep.Name(), path.Join(redoDir, tgt+DepSuffix))
		if err != nil {
			goto Finish
		}
		if !NoSync {
			err = syncDir(redoDir)
			if err != nil {
				goto Finish
			}
		}
	Finish:
		runErr.Err = err
		errs <- runErr
	}()
	return nil
}

func isOkRun(err error) bool {
	if err == nil {
		return true
	}
	if err, ok := err.(RunErr); ok && err.Err == nil {
		trace(CRedo, "%s", err.Name())
		return true
	}
	trace(CErr, "%s", err)
	return false
}
