/*
goredo -- djb's redo implementation on pure Go
Copyright (C) 2020-2023 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// Out-of-date determination

package main

import (
	"errors"
	"fmt"
	"io"
	"log"
	"os"
	"path"
	"path/filepath"
	"strings"

	"golang.org/x/sys/unix"
)

const (
	DepTypeIfcreate = "ifcreate"
	DepTypeIfchange = "ifchange"
	DepTypeAlways   = "always"
	DepTypeStamp    = "stamp"

	EnvOODTgtsFd     = "REDO_OOD_TGTS_FD"
	EnvOODTgtsLockFd = "REDO_OOD_TGTS_LOCK_FD"
)

var (
	OODTgts       map[string]struct{}
	FdOODTgts     *os.File
	FdOODTgtsLock *os.File

	OODCache        map[string]bool = make(map[string]bool)
	FileExistsCache map[string]bool = make(map[string]bool)

	ErrMissingTarget = errors.New("invalid format of .rec: missing Target")
)

func FileExists(p string) bool {
	if exists, known := FileExistsCache[p]; known {
		return exists
	}
	_, err := os.Stat(p)
	if err == nil {
		FileExistsCache[p] = true
		return true
	}
	if os.IsNotExist(err) {
		FileExistsCache[p] = false
	}
	return false
}

type TgtError struct {
	Tgt string
	Err error
}

func (e TgtError) Unwrap() error { return e.Err }

func (e TgtError) Error() string {
	return fmt.Sprintf("%s: %s", e.Tgt, e.Err)
}

func cwdMustRel(paths ...string) string {
	rel, err := filepath.Rel(Cwd, path.Join(paths...))
	if err != nil {
		panic(err)
	}
	return rel
}

func cwdAndTgt(tgt string) (string, string) {
	cwd, tgt := path.Split(tgt)
	cwd, err := filepath.Abs(cwd)
	if err != nil {
		panic(err)
	}
	return cwd, tgt
}

func isSrc(cwd, tgt string) bool {
	d, f := path.Split(path.Join(cwd, tgt))
	if !FileExists(path.Join(d, f)) {
		return false
	}
	if FileExists(path.Join(d, f+".do")) {
		return false
	}
	if FileExists(path.Join(d, RedoDir, f+DepSuffix)) {
		return false
	}
	return true
}

func isOODByBuildUUID(cwd, tgtOrig string) bool {
	cwd, tgt := cwdAndTgt(path.Join(cwd, tgtOrig))
	depPath := path.Join(cwd, RedoDir, tgt+DepSuffix)
	fdDep, err := os.Open(depPath)
	if err != nil {
		return true
	}
	depInfo, err := depRead(fdDep)
	fdDep.Close()
	if err != nil || depInfo.build != BuildUUID {
		return true
	}
	return false
}

func isOOD(cwd, tgtOrig string, level int, seen map[string]struct{}) (bool, error) {
	indent := strings.Repeat(". ", level)
	tracef(CDebug, "ood: %s%s checking", indent, tgtOrig)
	cwd, tgt := cwdAndTgt(path.Join(cwd, tgtOrig))
	ood, cached := OODCache[path.Join(cwd, tgt)]
	if cached {
		tracef(CDebug, "ood: %s%s -> cached: %v", indent, tgtOrig, ood)
		return ood, nil
	}
	depPath := path.Join(cwd, RedoDir, tgt+DepSuffix)
	fdDep, err := os.Open(depPath)
	if err != nil {
		if isSrc(cwd, tgt) {
			ood = false
			tracef(CDebug, "ood: %s%s -> is source", indent, tgtOrig)
		} else {
			ood = true
			tracef(CDebug, "ood: %s%s -> no dep: %s", indent, tgtOrig, depPath)
		}
		OODCache[path.Join(cwd, tgt)] = ood
		return ood, nil
	}
	depInfo, err := depRead(fdDep)
	fdDep.Close()
	if err != nil {
		return true, TgtError{tgtOrig, err}
	}

	if depInfo.build == BuildUUID {
		tracef(CDebug, "ood: %s%s -> already built", indent, tgtOrig)
		OODCache[path.Join(cwd, tgt)] = false
		return false, nil
	}
	if !FileExists(path.Join(cwd, tgt)) {
		tracef(CDebug, "ood: %s%s -> non-existent", indent, tgtOrig)
		OODCache[path.Join(cwd, tgt)] = true
		return true, nil
	}

	for _, dep := range depInfo.ifcreates {
		if FileExists(path.Join(cwd, dep)) {
			tracef(CDebug, "ood: %s%s -> %s created", indent, tgtOrig, dep)
			ood = true
			goto Done
		}
	}

	for _, m := range depInfo.ifchanges {
		dep := m["Target"]
		if dep == "" {
			return ood, TgtError{tgtOrig, ErrMissingTarget}
		}
		theirInode, err := inodeFromRec(m)
		if err != nil {
			return ood, TgtError{tgtOrig, fmt.Errorf("invalid format of .rec: %w", err)}
		}
		theirHsh := m["Hash"]
		tracef(CDebug, "ood: %s%s -> %s: checking", indent, tgtOrig, dep)
		ood, cached = OODCache[path.Join(cwd, dep)]
		if cached {
			tracef(CDebug, "ood: %s%s -> %s: cached: %v", indent, tgtOrig, dep, ood)
			if ood {
				goto Done
			}
			continue
		}

		inode, err := inodeFromFileByPath(path.Join(cwd, dep))
		if err != nil {
			if os.IsNotExist(err) {
				tracef(CDebug, "ood: %s%s -> %s: not exists", indent, tgtOrig, dep)
				ood = true
				OODCache[path.Join(cwd, dep)] = ood
				goto Done
			}
			return ood, TgtError{tgtOrig, err}
		}

		if inode.Size != theirInode.Size {
			tracef(CDebug, "ood: %s%s -> %s: size differs", indent, tgtOrig, dep)
			ood = true
			OODCache[path.Join(cwd, dep)] = ood
			goto Done
		}
		if InodeTrust != InodeTrustNone && inode.Equals(theirInode) {
			tracef(CDebug, "ood: %s%s -> %s: same inode", indent, tgtOrig, dep)
		} else {
			tracef(CDebug, "ood: %s%s -> %s: inode differs", indent, tgtOrig, dep)
			fd, err := os.Open(path.Join(cwd, dep))
			if err != nil {
				return ood, TgtError{tgtOrig, err}
			}
			hsh, err := fileHash(fd)
			fd.Close()
			if err != nil {
				return ood, TgtError{tgtOrig, err}
			}
			if theirHsh != hsh {
				tracef(CDebug, "ood: %s%s -> %s: hash differs", indent, tgtOrig, dep)
				ood = true
				OODCache[path.Join(cwd, dep)] = ood
				goto Done
			}
			tracef(CDebug, "ood: %s%s -> %s: same hash", indent, tgtOrig, dep)
		}

		if dep == tgt {
			tracef(CDebug, "ood: %s%s -> %s: same target", indent, tgtOrig, dep)
			continue
		}
		if isSrc(cwd, dep) {
			tracef(CDebug, "ood: %s%s -> %s: is source", indent, tgtOrig, dep)
			OODCache[path.Join(cwd, dep)] = false
			continue
		}

		if _, ok := seen[cwdMustRel(cwd, dep)]; ok {
			tracef(CDebug, "ood: %s%s -> %s: was always built", indent, tgtOrig, dep)
			OODCache[path.Join(cwd, dep)] = false
			continue
		}

		depOod, err := isOODWithTrace(cwd, dep, level+1, seen)
		if err != nil {
			return ood, TgtError{tgtOrig, err}
		}
		if depOod {
			tracef(CDebug, "ood: %s%s -> %s: ood", indent, tgtOrig, dep)
			ood = true
			goto Done
		}
		tracef(CDebug, "ood: %s%s -> %s: !ood", indent, tgtOrig, dep)
	}

Done:
	tracef(CDebug, "ood: %s%s: %v", indent, tgtOrig, ood)
	OODCache[path.Join(cwd, tgt)] = ood
	return ood, nil
}

func isOODWithTrace(
	cwd, tgtOrig string,
	level int,
	seen map[string]struct{},
) (bool, error) {
	p, err := filepath.Abs(path.Join(cwd, tgtOrig))
	if err != nil {
		panic(err)
	}
	_, ood := OODTgts[p]
	if ood {
		if !isOODByBuildUUID(cwd, tgtOrig) {
			tracef(
				CDebug,
				"ood: %s%s -> already built",
				strings.Repeat(". ", level), tgtOrig,
			)
			return false, nil
		}
		tracef(
			CDebug,
			"ood: %s%s true, external decision",
			strings.Repeat(". ", level), tgtOrig,
		)
		goto RecordOODTgt
	}
	ood, err = isOOD(cwd, tgtOrig, level, seen)
	if !ood {
		return ood, err
	}
RecordOODTgt:
	flock := unix.Flock_t{
		Type:   unix.F_WRLCK,
		Whence: io.SeekStart,
	}
	if err = unix.FcntlFlock(FdOODTgtsLock.Fd(), unix.F_SETLKW, &flock); err != nil {
		log.Fatalln(err)
	}
	if _, err = FdOODTgts.Seek(0, io.SeekEnd); err != nil {
		log.Fatalln(err)
	}
	if _, err := FdOODTgts.WriteString(p + "\x00"); err != nil {
		log.Fatalln(err)
	}
	flock.Type = unix.F_UNLCK
	if err = unix.FcntlFlock(FdOODTgtsLock.Fd(), unix.F_SETLK, &flock); err != nil {
		log.Fatalln(err)
	}
	return true, nil
}

func oodTgtsClear() {
	var err error
	flock := unix.Flock_t{
		Type:   unix.F_WRLCK,
		Whence: io.SeekStart,
	}
	if err = unix.FcntlFlock(FdOODTgtsLock.Fd(), unix.F_SETLKW, &flock); err != nil {
		log.Fatalln(err)
	}
	if err = FdOODTgts.Truncate(0); err != nil {
		log.Fatalln(err)
	}
	flock.Type = unix.F_UNLCK
	if err = unix.FcntlFlock(FdOODTgtsLock.Fd(), unix.F_SETLK, &flock); err != nil {
		log.Fatalln(err)
	}
}
