/*
goredo -- djb's redo implementation on pure Go
Copyright (C) 2020-2023 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

package main

import (
	"bufio"
	"bytes"
	"crypto/rand"
	"errors"
	"flag"
	"fmt"
	"io"
	"log"
	"os"
	"os/signal"
	"path"
	"path/filepath"
	"runtime"
	"sort"
	"strconv"
	"syscall"

	"go.cypherpunks.ru/recfile"
	"golang.org/x/sys/unix"
)

const (
	CmdNameGoredo       = "goredo"
	CmdNameRedo         = "redo"
	CmdNameRedoAffects  = "redo-affects"
	CmdNameRedoAlways   = "redo-always"
	CmdNameRedoCleanup  = "redo-cleanup"
	CmdNameRedoDot      = "redo-dot"
	CmdNameRedoIfchange = "redo-ifchange"
	CmdNameRedoIfcreate = "redo-ifcreate"
	CmdNameRedoLog      = "redo-log"
	CmdNameRedoOOD      = "redo-ood"
	CmdNameRedoSources  = "redo-sources"
	CmdNameRedoStamp    = "redo-stamp"
	CmdNameRedoTargets  = "redo-targets"
	CmdNameRedoWhichdo  = "redo-whichdo"
	CmdNameRedoDepFix   = "redo-depfix"
	CmdNameRedoInode    = "redo-inode"
)

var (
	Cwd       string
	BuildUUID string
	IsTopRedo bool // is it the top redo instance
	UmaskCur  int
)

func mustSetenv(key string) {
	if err := os.Setenv(key, "1"); err != nil {
		panic(err)
	}
}

func mustParseFd(v, name string) *os.File {
	ptr, err := strconv.ParseUint(v, 10, 64)
	if err != nil {
		panic(err)
	}
	fd := os.NewFile(uintptr(ptr), name)
	if fd == nil {
		panic("can not parse fd: " + name)
	}
	return fd
}

func CmdName() string {
	return path.Base(os.Args[0])
}

func main() {
	version := flag.Bool("version", false, "print version")
	warranty := flag.Bool("warranty", false, "print warranty information")

	var symlinks *bool
	cmdName := CmdName()
	if cmdName == "goredo" {
		symlinks = flag.Bool("symlinks", false, "create necessary symlinks in current directory")
	}

	flag.Usage = func() { usage(os.Args[0]) }
	flag.Parse()
	if *warranty {
		fmt.Println(Warranty)
		return
	}
	if *version {
		fmt.Println("goredo", Version, "built with", runtime.Version())
		return
	}
	if cmdName == CmdNameGoredo && *symlinks {
		rc := 0
		for _, cmdName := range []string{
			CmdNameRedo,
			CmdNameRedoAffects,
			CmdNameRedoAlways,
			CmdNameRedoCleanup,
			CmdNameRedoDepFix,
			CmdNameRedoDot,
			CmdNameRedoIfchange,
			CmdNameRedoIfcreate,
			CmdNameRedoLog,
			CmdNameRedoOOD,
			CmdNameRedoSources,
			CmdNameRedoStamp,
			CmdNameRedoTargets,
			CmdNameRedoWhichdo,
		} {
			fmt.Println(os.Args[0], "<-", cmdName)
			if err := os.Symlink(os.Args[0], cmdName); err != nil {
				rc = 1
				log.Println(err)
			}
		}
		os.Exit(rc)
	}
	log.SetFlags(log.Lshortfile)

	UmaskCur = syscall.Umask(0)
	syscall.Umask(UmaskCur)

	var err error
	Cwd, err = os.Getwd()
	if err != nil {
		log.Fatalln(err)
	}

	TopDir = os.Getenv(EnvTopDir)
	if TopDir == "" {
		TopDir = "/"
	} else {
		TopDir, err = filepath.Abs(TopDir)
		if err != nil {
			panic(err)
		}
	}
	DirPrefix = os.Getenv(EnvDirPrefix)
	DepCwd = os.Getenv(EnvDepCwd)

	if flagStderrKeep != nil && *flagStderrKeep {
		mustSetenv(EnvStderrKeep)
	}
	if flagStderrSilent != nil && *flagStderrSilent {
		mustSetenv(EnvStderrSilent)
	}
	if flagNoProgress != nil && *flagNoProgress {
		mustSetenv(EnvNoProgress)
	}
	if flagDebug != nil && *flagDebug {
		mustSetenv(EnvDebug)
	}
	if flagLogWait != nil && *flagLogWait {
		mustSetenv(EnvLogWait)
	}
	if flagLogLock != nil && *flagLogLock {
		mustSetenv(EnvLogLock)
	}
	if flagLogPid != nil && *flagLogPid {
		mustSetenv(EnvLogPid)
	}
	if flagLogJS != nil && *flagLogJS {
		mustSetenv(EnvLogJS)
	}
	StderrKeep = os.Getenv(EnvStderrKeep) == "1"
	StderrSilent = os.Getenv(EnvStderrSilent) == "1"
	NoProgress = os.Getenv(EnvNoProgress) == "1"
	Debug = os.Getenv(EnvDebug) == "1"
	LogWait = os.Getenv(EnvLogWait) == "1"
	LogLock = os.Getenv(EnvLogLock) == "1"
	LogJS = os.Getenv(EnvLogJS) == "1"
	if Debug || os.Getenv(EnvLogPid) == "1" {
		MyPid = os.Getpid()
	}
	var traced bool
	if flagTraceAll != nil && *flagTraceAll {
		mustSetenv(EnvTrace)
	}
	if os.Getenv(EnvTrace) == "1" {
		TracedAll = true
		traced = true
	} else if flagTrace != nil {
		traced = *flagTrace
	}
	NoColor = os.Getenv(EnvNoColor) != ""
	NoSync = os.Getenv(EnvNoSync) == "1"
	StopIfMod = os.Getenv(EnvStopIfMod) == "1"
	switch s := os.Getenv(EnvInodeTrust); s {
	case "none":
		InodeTrust = InodeTrustNone
	case "", "ctime":
		InodeTrust = InodeTrustCtime
	case "mtime":
		InodeTrust = InodeTrustMtime
	default:
		log.Fatalln("unknown", EnvInodeTrust, "value")
	}
	tracef(CDebug, "inode-trust: %s", InodeTrust)

	// Those are internal envs
	FdOODTgts, err = os.CreateTemp("", "ood-tgts")
	if err != nil {
		log.Fatalln(err)
	}
	if err = os.Remove(FdOODTgts.Name()); err != nil {
		log.Fatalln(err)
	}
	FdOODTgtsLock, err = os.CreateTemp("", "ood-tgts.lock")
	if err != nil {
		log.Fatalln(err)
	}
	if err = os.Remove(FdOODTgtsLock.Name()); err != nil {
		log.Fatalln(err)
	}

	if v := os.Getenv(EnvOODTgtsFd); v != "" {
		fd := mustParseFd(v, EnvOODTgtsFd)
		fdLock := mustParseFd(v, EnvOODTgtsLockFd)
		flock := unix.Flock_t{
			Type:   unix.F_WRLCK,
			Whence: io.SeekStart,
		}
		if err = unix.FcntlFlock(fdLock.Fd(), unix.F_SETLKW, &flock); err != nil {
			log.Fatalln(err)
		}
		if _, err = fd.Seek(0, io.SeekStart); err != nil {
			log.Fatalln(err)
		}
		tgtsRaw, err := io.ReadAll(bufio.NewReader(fd))
		if err != nil {
			log.Fatalln(err)
		}
		flock.Type = unix.F_UNLCK
		if err = unix.FcntlFlock(fdLock.Fd(), unix.F_SETLK, &flock); err != nil {
			log.Fatalln(err)
		}
		OODTgts = map[string]struct{}{}
		for _, tgtRaw := range bytes.Split(tgtsRaw, []byte{0}) {
			t := string(tgtRaw)
			if t == "" {
				continue
			}
			OODTgts[t] = struct{}{}
			tracef(CDebug, "ood: known to be: %s", t)
		}
	}

	StderrPrefix = os.Getenv(EnvStderrPrefix)
	if v := os.Getenv(EnvLevel); v != "" {
		Level, err = strconv.Atoi(v)
		if err != nil {
			panic(err)
		}
		if Level < 0 {
			panic("negative " + EnvLevel)
		}
	}

	var fdDep *os.File
	if v := os.Getenv(EnvDepFd); v != "" {
		fdDep = mustParseFd(v, EnvDepFd)
	}

	tgts := flag.Args()
	BuildUUID = os.Getenv(EnvBuildUUID)
	tgtsWasEmpty := len(tgts) == 0
	if BuildUUID == "" {
		IsTopRedo = true
		raw := new([16]byte)
		if _, err = io.ReadFull(rand.Reader, raw[:]); err != nil {
			log.Fatalln(err)
		}
		raw[6] = (raw[6] & 0x0F) | uint8(4<<4) // version 4
		BuildUUID = fmt.Sprintf(
			"%x-%x-%x-%x-%x",
			raw[0:4], raw[4:6], raw[6:8], raw[8:10], raw[10:],
		)
		if tgtsWasEmpty {
			tgts = []string{"all"}
		}
	}

	if cmdName == CmdNameRedo {
		statusInit()
	}

	for i, tgt := range tgts {
		if path.IsAbs(tgt) {
			tgts[i] = cwdMustRel(tgt)
		}
	}

	killed := make(chan os.Signal, 1)
	signal.Notify(killed, syscall.SIGHUP, syscall.SIGINT, syscall.SIGTERM)
	go func() {
		<-killed
		tracef(CDebug, "[%s] killed", BuildUUID)
		jsReleaseAll()
		RunningProcsM.Lock()
		for pid, proc := range RunningProcs {
			tracef(CDebug, "[%s] killing child %d", BuildUUID, pid)
			_ = proc.Signal(syscall.SIGTERM)
		}
		os.Exit(1)
	}()

	ok := true
	err = nil
	tracef(
		CDebug, "[%s] run: %s %s cwd:%s dirprefix:%s",
		BuildUUID, cmdName, tgts, Cwd, DirPrefix,
	)

CmdSwitch:
	switch cmdName {
	case CmdNameRedo:
		for _, tgt := range tgts {
			ok, err = ifchange([]string{tgt}, true, traced)
			if err != nil || !ok {
				break
			}
		}
	case CmdNameRedoIfchange:
		ok, err = ifchange(tgts, *flagForcedIfchange, traced)
		if err == nil {
			err = depsWrite(fdDep, tgts)
		}
	case CmdNameRedoIfcreate:
		if fdDep == nil {
			log.Fatalln("no", EnvDepFd)
		}
		for _, tgt := range tgts {
			tgtRel, err := filepath.Rel(
				path.Join(Cwd, DirPrefix),
				path.Join(Cwd, tgt),
			)
			if err != nil {
				break
			}
			err = ifcreate(fdDep, tgtRel)
			if err != nil {
				break
			}
		}
	case CmdNameRedoAlways:
		if fdDep == nil {
			log.Fatalln("no", EnvDepFd)
		}
		err = always(fdDep)
	case CmdNameRedoCleanup:
		for _, what := range tgts {
			err = cleanupWalker(Cwd, what)
			if err != nil {
				break
			}
		}
	case CmdNameRedoDot:
		err = dotPrint(tgts)
	case CmdNameRedoStamp:
		if fdDep == nil {
			log.Fatalln("no", EnvDepFd)
		}
		err = stamp(fdDep, os.Stdin)
	case CmdNameRedoLog:
		if len(tgts) != 1 {
			log.Fatalln("single target expected")
		}
		d, t := cwdAndTgt(tgts[0])
		err = showBuildLog(d, t, nil, 0)
	case CmdNameRedoWhichdo:
		if len(tgts) != 1 {
			log.Fatalln("single target expected")
		}
		var fdTmp *os.File
		fdTmp, err = os.CreateTemp("", "whichdo")
		if err != nil {
			break
		}
		if err = os.Remove(fdTmp.Name()); err != nil {
			break
		}
		cwd, tgt := cwdAndTgt(tgts[0])
		doFile, upLevels, err := findDo(fdTmp, cwd, tgt)
		if err != nil {
			break
		}
		_, err = fdTmp.Seek(0, io.SeekStart)
		if err != nil {
			break
		}
		r := recfile.NewReader(fdTmp)
		for {
			m, err := r.NextMap()
			if err != nil {
				if errors.Is(err, io.EOF) {
					break
				}
				break CmdSwitch
			}
			fmt.Println(cwdMustRel(cwd, m["Target"]))
		}
		if doFile == "" {
			ok = false
		} else {
			p := make([]string, 0, upLevels+2)
			p = append(p, cwd)
			for i := 0; i < upLevels; i++ {
				p = append(p, "..")
			}
			p = append(p, doFile)
			rel, err := filepath.Rel(Cwd, path.Join(p...))
			if err != nil {
				panic(err)
			}
			fmt.Println(rel)
		}
	case CmdNameRedoTargets:
		if tgtsWasEmpty {
			tgts = []string{Cwd}
		}
		tgts, err = targetsWalker(tgts)
		if err != nil {
			break
		}
		sort.Strings(tgts)
		for _, tgt := range tgts {
			fmt.Println(tgt)
		}
	case CmdNameRedoAffects:
		if tgtsWasEmpty {
			log.Fatalln("no targets specified")
		}
		var tgtsKnown []string
		tgtsKnown, err = targetsWalker([]string{Cwd})
		if err != nil {
			break
		}
		deps := map[string]map[string]struct{}{}
		for _, tgt := range tgtsKnown {
			collectDeps(Cwd, tgt, 0, deps, true)
		}
		seen := map[string]struct{}{}
		for _, tgt := range tgts {
			collectWholeDeps(deps[tgt], deps, seen)
		}
		tgts := make([]string, 0, len(seen))
		for dep := range seen {
			tgts = append(tgts, dep)
		}
		sort.Strings(tgts)
		for _, dep := range tgts {
			fmt.Println(dep)
		}
	case CmdNameRedoOOD:
		if tgtsWasEmpty {
			tgts, err = targetsWalker([]string{Cwd})
			if err != nil {
				break
			}
		}
		sort.Strings(tgts)
		var ood bool
		for _, tgt := range tgts {
			ood, err = isOOD(Cwd, tgt, 0, nil)
			if err != nil {
				break
			}
			if ood {
				fmt.Println(tgt)
			}
		}
	case CmdNameRedoSources:
		if tgtsWasEmpty {
			tgts, err = targetsWalker([]string{Cwd})
			if err != nil {
				break
			}
		}
		sort.Strings(tgts)
		var srcs []string
		srcs, err = sourcesWalker(tgts)
		sort.Strings(srcs)
		for _, src := range srcs {
			fmt.Println(src)
		}
	case CmdNameRedoDepFix:
		err = depFix(Cwd)
	case CmdNameRedoInode:
		var inode *Inode
		for _, tgt := range tgts {
			inode, err = inodeFromFileByPath(tgt)
			if err != nil {
				break
			}
			err = recfileWrite(os.Stdout, append(
				[]recfile.Field{{Name: "Target", Value: tgt}},
				inode.RecfileFields()...)...)
			if err != nil {
				break
			}
		}
	default:
		log.Fatalln("unknown command", cmdName)
	}
	if err != nil {
		log.Println(err)
	}
	rc := 0
	if !ok || err != nil {
		rc = 1
	}
	tracef(CDebug, "[%s] finished: %s %s", BuildUUID, cmdName, tgts)
	os.Exit(rc)
}
