/*
go.cypherpunks.ru/tai64n -- Pure Go TAI64/TAI64N implementation
Copyright (C) 2020-2023 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

package tai64n

import (
	"testing"
	"time"
)

func TestVector(t *testing.T) {
	tm, err := Decode("400000002a2b2c2d")
	if err != nil {
		t.Fatal(err)
	}

	ref := time.Date(1992, 6, 2, 8, 7, 9, 0, time.UTC).Add(-Leapsecs1972 * time.Second)
	if !tm.Equal(ref) {
		t.Fatal("TAI64 != reference")
	}

	tm = LeapsecsSub(tm)
	ref = time.Date(1992, 6, 2, 8, 6, 43, 0, time.UTC)
	if !tm.Equal(ref) {
		t.Fatal("UTC != reference")
	}
}

func BenchmarkTAI64(b *testing.B) {
	now := time.Now()
	now = time.Unix(now.Unix(), 0)
	tai := new(TAI64)
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		tai.FromTime(now)
		if !ToTime(tai[:]).Equal(now) {
			b.FailNow()
		}
	}
}

func BenchmarkTAI64N(b *testing.B) {
	now := time.Now()
	now = time.Unix(now.Unix(), now.UnixNano())
	tai := new(TAI64N)
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		tai.FromTime(now)
		if !ToTime(tai[:]).Equal(now) {
			b.FailNow()
		}
	}
}
