/*
goredo -- djb's redo implementation on pure Go
Copyright (C) 2020-2022 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// Targets runner

package main

import (
	"bufio"
	"errors"
	"flag"
	"fmt"
	"io"
	"log"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"strconv"
	"strings"
	"sync"
	"syscall"
	"time"

	"go.cypherpunks.ru/recfile"
	"go.cypherpunks.ru/tai64n/v2"
	"golang.org/x/sys/unix"
)

const (
	EnvDepFd        = "REDO_DEP_FD"
	EnvDirPrefix    = "REDO_DIRPREFIX"
	EnvDepCwd       = "REDO_DEP_CWD"
	EnvBuildUUID    = "REDO_BUILD_UUID"
	EnvStderrPrefix = "REDO_STDERR_PREFIX"
	EnvTrace        = "REDO_TRACE"
	EnvStderrKeep   = "REDO_LOGS"
	EnvStderrSilent = "REDO_SILENT"
	EnvNoSync       = "REDO_NO_SYNC"
	EnvStopIfMod    = "REDO_STOP_IF_MODIFIED"

	RedoDir      = ".redo"
	LockSuffix   = ".lock"
	DepSuffix    = ".rec"
	TmpPrefix    = ".redo."
	LogSuffix    = ".log"
	LogRecSuffix = ".log-rec"
)

var (
	NoSync       = false
	StderrKeep   = false
	StderrSilent = false
	StderrPrefix string
	StopIfMod    = false
	Jobs         sync.WaitGroup

	flagTrace          *bool
	flagTraceAll       *bool
	flagStderrKeep     *bool
	flagStderrSilent   *bool
	flagForcedIfchange *bool

	TracedAll bool

	RunningProcs  = map[int]*os.Process{}
	RunningProcsM sync.Mutex

	Err1WasTouched = errors.New("$1 was explicitly touched")
)

func init() {
	cmdName := CmdName()
	if !(cmdName == CmdNameRedo || cmdName == CmdNameRedoIfchange) {
		return
	}
	flagTrace = flag.Bool("x", false, "trace (sh -x) current targets")
	flagTraceAll = flag.Bool("xx", false,
		fmt.Sprintf("trace (sh -x) all targets (%s=1)", EnvTrace))
	flagStderrKeep = flag.Bool("k", false,
		fmt.Sprintf("keep job's stderr (%s=1)", EnvStderrKeep))
	flagStderrSilent = flag.Bool("s", false,
		fmt.Sprintf("silent, do not print job's stderr (%s=1)", EnvStderrSilent))
	flagForcedIfchange = flag.Bool("f", false, "forced redo-ifchange")
}

type RunError struct {
	Tgt      string
	DoFile   string
	Started  *time.Time
	Finished *time.Time
	Err      error
}

func (e *RunError) Name() string {
	var name string
	if e.DoFile == "" {
		name = e.Tgt
	} else {
		name = fmt.Sprintf("%s (%s)", e.Tgt, e.DoFile)
	}
	if e.Finished == nil {
		return name
	}
	return fmt.Sprintf("%s (%.3fs)", name, e.Finished.Sub(*e.Started).Seconds())
}

func (e RunError) Error() string {
	return fmt.Sprintf("%s: %s", e.Name(), e.Err)
}

func mkdirs(pth string) error {
	if _, err := os.Stat(pth); err == nil {
		return nil
	}
	return os.MkdirAll(pth, os.FileMode(0777))
}

func isModified(cwd, redoDir, tgt string) (bool, *Inode, string, error) {
	fdDep, err := os.Open(path.Join(redoDir, tgt+DepSuffix))
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil, "", nil
		}
		return false, nil, "", err
	}
	defer fdDep.Close()
	r := recfile.NewReader(fdDep)
	var modified bool
	var ourInode *Inode
	var hshPrev string
	for {
		m, err := r.NextMap()
		if err != nil {
			if errors.Is(err, io.EOF) {
				break
			}
			return false, nil, "", err
		}
		if m["Type"] != DepTypeIfchange || m["Target"] != tgt {
			continue
		}
		fd, err := os.Open(path.Join(cwd, tgt))
		if err != nil {
			if os.IsNotExist(err) {
				return false, nil, "", nil
			}
			return false, nil, "", err
		}
		ourInode, err = inodeFromFile(fd)
		fd.Close()
		if err != nil {
			return false, nil, "", err
		}
		theirInode, err := inodeFromRec(m)
		if err != nil {
			return false, nil, "", err
		}
		hshPrev = m["Hash"]
		modified = !ourInode.Equals(theirInode)
		break
	}
	return modified, ourInode, hshPrev, nil
}

func syncDir(dir string) error {
	fd, err := os.Open(dir)
	if err != nil {
		return err
	}
	err = fd.Sync()
	fd.Close()
	return err
}

func runScript(tgtOrig string, errs chan error, traced bool) error {
	cwd, tgt := cwdAndTgt(tgtOrig)
	redoDir := path.Join(cwd, RedoDir)
	if err := mkdirs(redoDir); err != nil {
		return TgtError{tgtOrig, err}
	}

	shCtx := fmt.Sprintf("sh: %s: cwd:%s", tgtOrig, cwd)
	jsToken := jsAcquire(shCtx)
	jsNeedsRelease := true
	defer func() {
		if jsNeedsRelease {
			jsRelease(shCtx, jsToken)
		}
	}()

	// Acquire lock
	fdLock, err := os.OpenFile(
		path.Join(redoDir, tgt+LockSuffix),
		os.O_WRONLY|os.O_TRUNC|os.O_CREATE,
		os.FileMode(0666),
	)
	if err != nil {
		return TgtError{tgtOrig, err}
	}
	flock := unix.Flock_t{
		Type:   unix.F_WRLCK,
		Whence: io.SeekStart,
	}
	lockRelease := func() {
		tracef(CLock, "LOCK_UN: %s", fdLock.Name())
		flock.Type = unix.F_UNLCK
		if err := unix.FcntlFlock(fdLock.Fd(), unix.F_SETLK, &flock); err != nil {
			log.Fatalln(err)
		}
		fdLock.Close()
	}
	tracef(CLock, "LOCK_NB: %s", fdLock.Name())

	// Waiting for job completion, already taken by someone else
	if err = unix.FcntlFlock(fdLock.Fd(), unix.F_SETLK, &flock); err != nil {
		if uintptr(err.(syscall.Errno)) != uintptr(unix.EAGAIN) {
			fdLock.Close()
			return TgtError{tgtOrig, err}
		}
		Jobs.Add(1)
		if err = unix.FcntlFlock(fdLock.Fd(), unix.F_GETLK, &flock); err != nil {
			log.Fatalln(err)
		}
		tracef(CDebug, "waiting: %s (pid=%d)", tgtOrig, flock.Pid)
		if FdStatus != nil {
			if _, err = FdStatus.Write([]byte{StatusWait}); err != nil {
				log.Fatalln(err)
			}
		}
		go func() {
			defer Jobs.Done()
			tracef(CLock, "LOCK_EX: %s", fdLock.Name())
			if err := unix.FcntlFlock(fdLock.Fd(), unix.F_SETLKW, &flock); err != nil {
				log.Fatalln(err)
			}
			lockRelease()
			tracef(CDebug, "waiting done: %s", tgtOrig)
			if FdStatus != nil {
				if _, err = FdStatus.Write([]byte{StatusWaited}); err != nil {
					log.Fatalln(err)
				}
			}
			var depInfo *DepInfo
			fdDep, err := os.Open(path.Join(redoDir, tgt+DepSuffix))
			if err != nil {
				if os.IsNotExist(err) {
					err = errors.New("was not built: no .rec")
				}
				goto Finish
			}
			defer fdDep.Close()
			depInfo, err = depRead(fdDep)
			if err != nil {
				goto Finish
			}
			if depInfo.build != BuildUUID {
				err = errors.New("was not built: build differs")
			}
		Finish:
			if err != nil {
				err = TgtError{tgtOrig, err}
				fdLock.Close()
			}
			errs <- err
		}()
		return nil
	}

	// Check if target is not modified externally
	modified, inodePrev, hshPrev, err := isModified(cwd, redoDir, tgt)
	if err != nil {
		lockRelease()
		return TgtError{tgtOrig, err}
	}
	if modified {
		lockRelease()
		if StopIfMod {
			return fmt.Errorf("%s externally modified", tgtOrig)
		}
		tracef(CWarn, "%s externally modified: not redoing", tgtOrig)
		go func() {
			errs <- nil
		}()
		return nil
	}

	// Start preparing .rec
	fdDep, err := tempfile(redoDir, tgt+DepSuffix)
	if err != nil {
		lockRelease()
		return TgtError{tgtOrig, err}
	}
	fdDepPath := fdDep.Name()
	cleanup := func() {
		lockRelease()
		fdDep.Close()
		os.Remove(fdDep.Name())
	}
	if _, err = recfile.NewWriter(fdDep).WriteFields(
		recfile.Field{Name: "Build", Value: BuildUUID},
	); err != nil {
		cleanup()
		return TgtError{tgtOrig, err}
	}

	// Find .do
	doFile, upLevels, err := findDo(fdDep, cwd, tgt)
	if err != nil {
		cleanup()
		return TgtError{tgtOrig, err}
	}
	if doFile == "" {
		cleanup()
		return TgtError{tgtOrig, errors.New("no .do found")}
	}

	// Determine basename and DIRPREFIX
	doFileRelPath := doFile
	ents := strings.Split(cwd, "/")
	ents = ents[len(ents)-upLevels:]
	dirPrefix := path.Join(ents...)
	cwdOrig := cwd
	for i := 0; i < upLevels; i++ {
		cwd = path.Join(cwd, "..")
		doFileRelPath = path.Join("..", doFileRelPath)
	}
	cwd = path.Clean(cwd)
	doFilePath := path.Join(cwd, doFile)
	basename := tgt
	runErr := RunError{Tgt: tgtOrig}
	if strings.HasPrefix(doFile, "default.") {
		basename = tgt[:len(tgt)-(len(doFile)-len("default.")-len(".do"))-1]
		runErr.DoFile = doFileRelPath
	}

	if err = depWrite(fdDep, cwdOrig, doFileRelPath, ""); err != nil {
		cleanup()
		return TgtError{tgtOrig, err}
	}
	fdDep.Close()
	tracef(CWait, "%s", runErr.Name())

	// Prepare command line
	var cmdName string
	var args []string
	if err = unix.Access(doFilePath, unix.X_OK); err == nil {
		cmdName = doFilePath
		args = make([]string, 0, 3)
	} else {
		cmdName = "/bin/sh"
		if traced || TracedAll {
			args = append(args, "-ex")
		} else {
			args = append(args, "-e")
		}
		args = append(args, doFile)
	}

	// Temporary file for stdout
	fdStdout, err := tempfile(cwdOrig, tgt)
	if err != nil {
		cleanup()
		return TgtError{tgtOrig, err}
	}
	stdoutPath := fdStdout.Name()
	fdStdout.Close()
	tmpPath := stdoutPath + ".3" // and for $3
	tmpPathRel, err := filepath.Rel(cwd, tmpPath)
	if err != nil {
		panic(err)
	}
	args = append(
		args,
		path.Join(dirPrefix, tgt),
		path.Join(dirPrefix, basename),
		tmpPathRel,
	)
	shCtx = fmt.Sprintf(
		"sh: %s: %s %s cwd:%s dirprefix:%s",
		tgtOrig, cmdName, args, cwd, dirPrefix,
	)

	cmd := exec.Command(cmdName, args...)
	cmd.Dir = cwd
	// cmd.Stdin reads from /dev/null by default
	cmd.Env = append(os.Environ(), fmt.Sprintf("%s=%d", EnvLevel, Level+1))
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", EnvDirPrefix, dirPrefix))
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", EnvBuildUUID, BuildUUID))

	childStderrPrefix := tempsuffix()
	cmd.Env = append(cmd.Env, fmt.Sprintf(
		"%s=%s", EnvStderrPrefix, childStderrPrefix,
	))

	fdNum := 0
	cmd.ExtraFiles = append(cmd.ExtraFiles, FdOODTgts)
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvOODTgtsFd, 3+fdNum))
	fdNum++
	cmd.ExtraFiles = append(cmd.ExtraFiles, FdOODTgtsLock)
	cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvOODTgtsLockFd, 3+fdNum))
	fdNum++

	if FdStatus == nil {
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=NO", EnvStatusFd))
	} else {
		cmd.ExtraFiles = append(cmd.ExtraFiles, FdStatus)
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvStatusFd, 3+fdNum))
		fdNum++
	}

	// Preparing stderr
	var fdStderr *os.File
	if StderrKeep {
		fdStderr, err = os.OpenFile(
			path.Join(redoDir, tgt+LogSuffix),
			os.O_WRONLY|os.O_CREATE|os.O_TRUNC,
			os.FileMode(0666),
		)
		if err != nil {
			cleanup()
			return TgtError{tgtOrig, err}
		}
	}
	tracef(CDebug, "%s", shCtx)

	jsNeedsRelease = false
	Jobs.Add(1)
	go func() {
		if JSR == nil {
			// infinite jobs
			cmd.Env = append(cmd.Env, fmt.Sprintf("%s=NO", EnvJobs))
		} else {
			cmd.ExtraFiles = append(cmd.ExtraFiles, JSR)
			cmd.ExtraFiles = append(cmd.ExtraFiles, JSW)
			makeFlags := fmt.Sprintf(
				"%s %s%d,%d", MakeFlags, MakeJSArg, 3+fdNum+0, 3+fdNum+1,
			)
			makeFlags = strings.Trim(makeFlags, " ")
			cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", MakeFlagsName, makeFlags))
			fdNum += 2
			cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvJSToken, jsToken))
		}

		if FdStatus != nil {
			if _, err = FdStatus.Write([]byte{StatusRun}); err != nil {
				log.Fatalln(err)
			}
		}

		defer jsRelease(shCtx, jsToken)

		var finished time.Time
		var exitErr *exec.ExitError
		started := time.Now()
		runErr.Started = &started
		fdStdout, err = os.OpenFile(stdoutPath, os.O_RDWR, os.FileMode(0666))
		if err != nil {
			if fdStderr != nil {
				fdStderr.Close()
			}
			runErr.Err = err
			errs <- runErr
			return
		}
		cmd.Stdout = fdStdout
		fdDep, err = os.OpenFile(fdDepPath, os.O_WRONLY|os.O_APPEND, os.FileMode(0666))
		if err != nil {
			if fdStderr != nil {
				fdStderr.Close()
			}
			fdStdout.Close()
			runErr.Err = err
			errs <- runErr
			return
		}
		cmd.ExtraFiles = append(cmd.ExtraFiles, fdDep)
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%d", EnvDepFd, 3+fdNum))
		fdNum++
		cmd.Env = append(cmd.Env, fmt.Sprintf("%s=%s", EnvDepCwd, cwd))

		defer func() {
			fdDep.Close()
			fdStdout.Close()
			if fdStderr != nil {
				fdStderr.Close()
				logRecPath := path.Join(redoDir, tgt+LogRecSuffix)
				if fdStderr, err = os.OpenFile(
					logRecPath,
					os.O_WRONLY|os.O_CREATE|os.O_TRUNC,
					os.FileMode(0666),
				); err == nil {
					fields := []recfile.Field{
						{Name: "Build", Value: BuildUUID},
						{Name: "PPID", Value: strconv.Itoa(os.Getpid())},
						{Name: "Cwd", Value: cwd},
					}
					if cmd.Process != nil {
						fields = append(fields, recfile.Field{
							Name: "PID", Value: strconv.Itoa(cmd.Process.Pid),
						})
					}
					ts := new(tai64n.TAI64N)
					ts.FromTime(started)
					fields = append(fields,
						recfile.Field{Name: "Started", Value: tai64n.Encode(ts[:])},
					)
					ts.FromTime(finished)
					fields = append(fields,
						recfile.Field{Name: "Finished", Value: tai64n.Encode(ts[:])})
					fields = append(fields, recfile.Field{
						Name:  "Duration",
						Value: strconv.FormatInt(finished.Sub(started).Nanoseconds(), 10),
					})
					fields = append(fields, recfile.Field{Name: "Cmd", Value: cmdName})
					for _, arg := range args {
						fields = append(fields, recfile.Field{Name: "Arg", Value: arg})
					}
					for _, env := range cmd.Env {
						fields = append(fields, recfile.Field{Name: "Env", Value: env})
					}
					if exitErr != nil {
						fields = append(fields, recfile.Field{
							Name:  "ExitCode",
							Value: strconv.Itoa(exitErr.ProcessState.ExitCode()),
						})
					}
					w := bufio.NewWriter(fdStderr)

					var depInfo *DepInfo
					fdDep, err := os.Open(fdDepPath)
					if err != nil {
						goto Err
					}
					depInfo, err = depRead(fdDep)
					fdDep.Close()
					if err != nil {
						goto Err
					}
					for _, dep := range depInfo.ifchanges {
						fields = append(fields, recfile.Field{
							Name:  "Ifchange",
							Value: dep["Target"],
						})
					}
					_, err = recfile.NewWriter(w).WriteFields(fields...)
					if err != nil {
						goto Err
					}
					err = w.Flush()
				Err:
					if err != nil {
						log.Println(err)
						os.Remove(logRecPath)
					}
					fdStderr.Close()
				} else {
					log.Println("can not open", logRecPath, ":", err)
				}
			}
			lockRelease()
			os.Remove(fdDep.Name())
			os.Remove(fdStdout.Name())
			os.Remove(tmpPath)
			os.Remove(fdLock.Name())
			if FdStatus != nil {
				if _, err = FdStatus.Write([]byte{StatusDone}); err != nil {
					log.Fatalln(err)
				}
			}
			Jobs.Done()
		}()
		stderr, err := cmd.StderrPipe()
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		started = time.Now()
		err = cmd.Start()
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		RunningProcsM.Lock()
		RunningProcs[cmd.Process.Pid] = cmd.Process
		RunningProcsM.Unlock()
		pid := fmt.Sprintf("[%d]", cmd.Process.Pid)
		tracef(CDebug, "%s runs %s", tgtOrig, pid)

		stderrTerm := make(chan struct{})
		go func() {
			scanner := bufio.NewScanner(stderr)
			var line string
			ts := new(tai64n.TAI64N)
			for scanner.Scan() {
				line = scanner.Text()
				if strings.HasPrefix(line, childStderrPrefix) {
					line = line[len(childStderrPrefix):]
					os.Stderr.WriteString(StderrPrefix + line + "\n")
					continue
				}
				if fdStderr != nil {
					ts.FromTime(time.Now())
					LogMutex.Lock()
					fmt.Fprintln(fdStderr, tai64n.Encode(ts[:]), line)
					LogMutex.Unlock()
				}
				if StderrSilent {
					continue
				}
				if MyPid == 0 {
					tracef(CNone, "%s", line)
				} else {
					tracef(CNone, "%s %s", pid, line)
				}
			}
			close(stderrTerm)
		}()

		// Wait for job completion
		<-stderrTerm
		err = cmd.Wait()
		RunningProcsM.Lock()
		delete(RunningProcs, cmd.Process.Pid)
		RunningProcsM.Unlock()
		finished = time.Now()
		runErr.Finished = &finished
		if err != nil {
			exitErr = err.(*exec.ExitError)
			runErr.Err = err
			errs <- runErr
			return
		}

		// Was $1 touched?
		if fd, err := os.Open(path.Join(cwdOrig, tgt)); err == nil {
			if inodePrev == nil {
				fd.Close()
				runErr.Err = Err1WasTouched
				errs <- runErr
				return
			}
			inode, err := inodeFromFile(fd)
			fd.Close()
			if err != nil {
				runErr.Err = err
				errs <- runErr
				return
			}
			if !inode.Equals(inodePrev) {
				runErr.Err = Err1WasTouched
				errs <- runErr
				return
			}
		}

		if inodePrev != nil {
			if fd, err := os.Open(path.Join(cwdOrig, tgt)); err == nil {
				inode, err := inodeFromFile(fd)
				fd.Close()
				if err == nil && !inode.Equals(inodePrev) {
					runErr.Err = Err1WasTouched
					errs <- runErr
					return
				}
			}
		}

		// Does it produce both stdout and tmp?
		fiStdout, err := os.Stat(fdStdout.Name())
		if err != nil {
			runErr.Err = err
			errs <- runErr
			return
		}
		tmpExists := false
		_, err = os.Stat(tmpPath)
		if err == nil {
			if fiStdout.Size() > 0 {
				runErr.Err = errors.New("created both tmp and stdout")
				errs <- runErr
				return
			}
			tmpExists = true
		} else if !os.IsNotExist(err) {
			runErr.Err = err
			errs <- runErr
			return
		}

		// Determine what file we must process at last
		var fd *os.File
		if tmpExists {
			fd, err = os.Open(tmpPath)
			if err != nil {
				goto Finish
			}
			defer fd.Close()
		} else if fiStdout.Size() > 0 {
			fd = fdStdout
		}

		// Do we need to ifcreate it, or ifchange with renaming?
		if fd == nil {
			os.Remove(path.Join(cwdOrig, tgt))
			err = ifcreate(fdDep, tgt)
			if err != nil {
				goto Finish
			}
		} else {
			var hsh string
			if hshPrev != "" {
				_, err = fd.Seek(0, io.SeekStart)
				if err != nil {
					goto Finish
				}
				hsh, err = fileHash(fd)
				if err != nil {
					goto Finish
				}
				if hsh == hshPrev {
					tracef(CDebug, "%s has same hash, not renaming", tgtOrig)
					err = os.Remove(fd.Name())
					if err != nil {
						goto Finish
					}
					err = os.Chtimes(path.Join(cwdOrig, tgt), finished, finished)
					if err != nil {
						goto Finish
					}
					if !NoSync {
						err = syncDir(cwdOrig)
						if err != nil {
							goto Finish
						}
					}
					err = depWrite(fdDep, cwdOrig, tgt, hshPrev)
					if err != nil {
						goto Finish
					}
					goto RecCommit
				}
			}
			if !NoSync {
				err = fd.Sync()
				if err != nil {
					goto Finish
				}
			}
			err = os.Rename(fd.Name(), path.Join(cwdOrig, tgt))
			if err != nil {
				goto Finish
			}
			if !NoSync {
				err = syncDir(cwdOrig)
				if err != nil {
					goto Finish
				}
			}
			err = depWrite(fdDep, cwdOrig, tgt, hsh)
			if err != nil {
				goto Finish
			}
		}

	RecCommit:
		// Commit .rec
		if !NoSync {
			err = fdDep.Sync()
			if err != nil {
				goto Finish
			}
		}
		fdDepPath = path.Join(redoDir, tgt+DepSuffix)
		err = os.Rename(fdDep.Name(), fdDepPath)
		if err != nil {
			goto Finish
		}
		if !NoSync {
			err = syncDir(redoDir)
			if err != nil {
				goto Finish
			}
		}

		// Post-commit .rec sanitizing
		fdDep.Close()
		if fdDepR, err := os.Open(fdDepPath); err == nil {
			depInfo, err := depRead(fdDepR)
			fdDepR.Close()
			if err != nil {
				goto Finish
			}
			ifchangeSeen := make(map[string]struct{}, len(depInfo.ifchanges))
			for _, dep := range depInfo.ifchanges {
				ifchangeSeen[dep["Target"]] = struct{}{}
			}
			for _, dep := range depInfo.ifcreates {
				if _, exists := ifchangeSeen[dep]; exists {
					tracef(CWarn, "simultaneous ifcreate and ifchange records: %s", tgt)
				}
			}
		}

	Finish:
		runErr.Err = err
		errs <- runErr
	}()
	return nil
}

func isOkRun(err error) bool {
	if err == nil {
		return true
	}
	var runErr RunError
	if errors.As(err, &runErr) && runErr.Err == nil {
		tracef(CRedo, "%s", runErr.Name())
		return true
	}
	tracef(CErr, "%s", err)
	return false
}
