/*
goredo -- djb's redo implementation on pure Go
Copyright (C) 2020-2022 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// Dependencies saver

package main

import (
	"bufio"
	"encoding/hex"
	"errors"
	"io"
	"os"
	"path"
	"path/filepath"

	"go.cypherpunks.ru/recfile"
	"lukechampine.com/blake3"
)

var (
	DirPrefix string
	DepCwd    string

	ErrBadRecFormat = errors.New("invalid format of .rec")
)

func recfileWrite(fdDep io.StringWriter, fields ...recfile.Field) error {
	w := recfile.NewWriter(fdDep)
	if _, err := w.RecordStart(); err != nil {
		return err
	}
	if _, err := w.WriteFields(fields...); err != nil {
		return err
	}
	return nil
}

func ifcreate(fdDep *os.File, tgt string) error {
	tracef(CDebug, "ifcreate: %s <- %s", fdDep.Name(), tgt)
	return recfileWrite(
		fdDep,
		recfile.Field{Name: "Type", Value: DepTypeIfcreate},
		recfile.Field{Name: "Target", Value: tgt},
	)
}

func always(fdDep *os.File) error {
	tracef(CDebug, "always: %s", fdDep.Name())
	return recfileWrite(fdDep, recfile.Field{Name: "Type", Value: DepTypeAlways})
}

func stamp(fdDep, src *os.File) error {
	var hsh string
	hsh, err := fileHash(src)
	if err != nil {
		return err
	}
	tracef(CDebug, "stamp: %s <- %s", fdDep.Name(), hsh)
	return recfileWrite(
		fdDep,
		recfile.Field{Name: "Type", Value: DepTypeStamp},
		recfile.Field{Name: "Hash", Value: hsh},
	)
}

func fileHash(fd *os.File) (string, error) {
	h := blake3.New(32, nil)
	if _, err := io.Copy(h, bufio.NewReader(fd)); err != nil {
		return "", err
	}
	return hex.EncodeToString(h.Sum(nil)), nil
}

func depWrite(fdDep *os.File, cwd, tgt string) error {
	tracef(CDebug, "ifchange: %s <- %s", fdDep.Name(), tgt)
	fd, err := os.Open(path.Join(cwd, tgt))
	if err != nil {
		return err
	}
	defer fd.Close()
	fi, err := fd.Stat()
	if err != nil {
		return err
	}
	if fi.IsDir() {
		return nil
	}
	inode, err := inodeFromFile(fd)
	if err != nil {
		return err
	}
	hsh, err := fileHash(fd)
	if err != nil {
		return err
	}
	fields := []recfile.Field{
		{Name: "Type", Value: DepTypeIfchange},
		{Name: "Target", Value: tgt},
		{Name: "Hash", Value: hsh},
	}
	fields = append(fields, inode.RecfileFields()...)
	return recfileWrite(fdDep, fields...)
}

func depsWrite(fdDep *os.File, tgts []string) error {
	if fdDep == nil {
		tracef(CDebug, "no opened fdDep: %s", tgts)
		return nil
	}
	for _, tgt := range tgts {
		tgtAbs, err := filepath.Abs(tgt)
		if err != nil {
			panic(err)
		}
		cwd := Cwd
		if DepCwd != "" && Cwd != DepCwd {
			cwd = DepCwd
		}
		tgtDir := path.Join(cwd, DirPrefix)
		tgtRel, err := filepath.Rel(tgtDir, tgtAbs)
		if err != nil {
			panic(err)
		}
		if _, errStat := os.Stat(tgt); errStat == nil {
			err = depWrite(fdDep, tgtDir, tgtRel)
		} else {
			tracef(CDebug, "ifchange: %s <- %s (non-existing)", fdDep.Name(), tgtRel)
			fields := []recfile.Field{
				{Name: "Type", Value: DepTypeIfchange},
				{Name: "Target", Value: tgtRel},
			}
			inodeDummy := Inode{}
			fields = append(fields, inodeDummy.RecfileFields()...)
			err = recfileWrite(fdDep, fields...)
		}
		if err != nil {
			return err
		}
	}
	return nil
}

type DepInfo struct {
	build     string
	always    bool
	stamp     string
	ifcreates []string
	ifchanges []map[string]string
}

func depRead(fdDep io.Reader) (*DepInfo, error) {
	r := recfile.NewReader(fdDep)
	m, err := r.NextMap()
	if err != nil {
		return nil, err
	}
	depInfo := DepInfo{}
	b := m["Build"]
	if b == "" {
		return nil, errors.New(".rec missing Build:")
	}
	depInfo.build = b
	for {
		m, err := r.NextMap()
		if err != nil {
			if errors.Is(err, io.EOF) {
				break
			}
			return nil, err
		}
		switch m["Type"] {
		case DepTypeAlways:
			depInfo.always = true
		case DepTypeIfcreate:
			dep := m["Target"]
			if dep == "" {
				return nil, ErrBadRecFormat
			}
			depInfo.ifcreates = append(depInfo.ifcreates, dep)
		case DepTypeIfchange:
			delete(m, "Type")
			depInfo.ifchanges = append(depInfo.ifchanges, m)
		case DepTypeStamp:
			hsh := m["Hash"]
			if hsh == "" {
				return nil, ErrBadRecFormat
			}
			depInfo.stamp = hsh
		default:
			return nil, ErrBadRecFormat
		}
	}
	return &depInfo, nil
}
