/*
goredo -- redo implementation on pure Go
Copyright (C) 2020-2021 Sergey Matveev <stargrave@stargrave.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

// Dependencies saver

package main

import (
	"bufio"
	"encoding/hex"
	"errors"
	"fmt"
	"io"
	"os"
	"path"
	"path/filepath"

	"go.cypherpunks.ru/recfile"
	"golang.org/x/sys/unix"
	"lukechampine.com/blake3"
)

var DirPrefix string

func recfileWrite(fdDep *os.File, fields ...recfile.Field) error {
	w := recfile.NewWriter(fdDep)
	if _, err := w.RecordStart(); err != nil {
		return err
	}
	if _, err := w.WriteFields(fields...); err != nil {
		return err
	}
	return nil
}

func ifcreate(fdDep *os.File, tgt string) error {
	trace(CDebug, "ifcreate: %s <- %s", fdDep.Name(), tgt)
	return recfileWrite(
		fdDep,
		recfile.Field{Name: "Type", Value: DepTypeIfcreate},
		recfile.Field{Name: "Target", Value: tgt},
	)
}

func always(fdDep *os.File) error {
	trace(CDebug, "always: %s", fdDep.Name())
	return recfileWrite(fdDep, recfile.Field{Name: "Type", Value: DepTypeAlways})
}

func stamp(fdDep, src *os.File) error {
	var hsh string
	hsh, err := fileHash(os.Stdin)
	if err != nil {
		return err
	}
	trace(CDebug, "stamp: %s <- %s", fdDep.Name(), hsh)
	return recfileWrite(
		fdDep,
		recfile.Field{Name: "Type", Value: DepTypeStamp},
		recfile.Field{Name: "Hash", Value: hsh},
	)
}

func fileCtime(fd *os.File) (string, error) {
	var stat unix.Stat_t
	if err := unix.Fstat(int(fd.Fd()), &stat); err != nil {
		return "", err
	}
	sec, nsec := stat.Ctim.Unix()
	return fmt.Sprintf("%d.%d", sec, nsec), nil
}

func fileHash(fd *os.File) (string, error) {
	h := blake3.New(32, nil)
	if _, err := io.Copy(h, bufio.NewReader(fd)); err != nil {
		return "", err
	}
	return hex.EncodeToString(h.Sum(nil)), nil
}

func writeDep(fdDep *os.File, cwd, tgt string) error {
	trace(CDebug, "ifchange: %s <- %s", fdDep.Name(), tgt)
	fd, err := os.Open(path.Join(cwd, tgt))
	if err != nil {
		return err
	}
	defer fd.Close()
	fi, err := fd.Stat()
	if err != nil {
		return err
	}
	if fi.IsDir() {
		return nil
	}
	ts, err := fileCtime(fd)
	if err != nil {
		return err
	}
	hsh, err := fileHash(fd)
	if err != nil {
		return err
	}
	return recfileWrite(
		fdDep,
		recfile.Field{Name: "Type", Value: DepTypeIfchange},
		recfile.Field{Name: "Target", Value: tgt},
		recfile.Field{Name: "Ctime", Value: ts},
		recfile.Field{Name: "Hash", Value: hsh},
	)
}

func writeDeps(fdDep *os.File, tgts []string) error {
	if fdDep == nil {
		trace(CDebug, "no opened fdDep: %s", tgts)
		return nil
	}
	for _, tgt := range tgts {
		if _, err := os.Stat(tgt); err == nil {
			tgtAbs, err := filepath.Abs(tgt)
			if err != nil {
				panic(err)
			}
			tgtDir := path.Join(Cwd, DirPrefix)
			tgtRel, err := filepath.Rel(tgtDir, tgtAbs)
			if err != nil {
				panic(err)
			}
			if err = writeDep(fdDep, tgtDir, tgtRel); err != nil {
				return err
			}
		} else {
			trace(CDebug, "skipping dep record, can not stat: %s", tgt)
		}
	}
	return nil
}

type DepInfo struct {
	build     string
	always    bool
	stamp     string
	ifcreates []string
	ifchanges []map[string]string
}

func depRead(fdDep *os.File) (*DepInfo, error) {
	r := recfile.NewReader(fdDep)
	m, err := r.NextMap()
	if err != nil {
		return nil, err
	}
	depInfo := DepInfo{}
	if b := m["Build"]; b == "" {
		return nil, errors.New(".dep missing Build:")
	} else {
		depInfo.build = b
	}
	for {
		m, err := r.NextMap()
		if err != nil {
			if err == io.EOF {
				break
			}
			return nil, err
		}
		switch m["Type"] {
		case DepTypeAlways:
			depInfo.always = true
		case DepTypeIfcreate:
			dep := m["Target"]
			if dep == "" {
				return nil, errors.New("invalid format of .dep")
			}
			depInfo.ifcreates = append(depInfo.ifcreates, dep)
		case DepTypeIfchange:
			delete(m, "Type")
			depInfo.ifchanges = append(depInfo.ifchanges, m)
		case DepTypeStamp:
			hsh := m["Hash"]
			if hsh == "" {
				return nil, errors.New("invalid format of .dep")
			}
			depInfo.stamp = hsh
		default:
			return nil, errors.New("invalid format of .dep")
		}
	}
	return &depInfo, nil
}
